/**
 * OWASP AppSensor
 * 
 * This file is part of the Open Web Application Security Project (OWASP)
 * AppSensor project. For details, please see
 * <a href="http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project">
 * 	http://www.owasp.org/index.php/Category:OWASP_AppSensor_Project</a>.
 *
 * Copyright (c) 2010 - The OWASP Foundation
 * 
 * AppSensor is published by OWASP under the BSD license. You should read and accept the
 * LICENSE before you use, modify, and/or redistribute this software.
 * 
 * @author Michael Coates <a href="http://www.aspectsecurity.com">Aspect Security</a>
 * @author John Melton <a href="http://www.jtmelton.com/">jtmelton</a>
 * @created 2010
 */
namespace org.owasp.appsensor
{
    using System;
    using System.Collections.Generic;
    using System.Threading;
    using org.owasp.appsensor.intrusiondetection;
    using AppSensor2.configuration;

    /**
     * The AppSensorServiceController houses the service monitors for all of the 
     * accessed resources/components in the application.  By default the service
     * is considered the request URI associated to a user request.  
     * <p>
     * The intention is that there is management of both an "all users" table that 
     * controls whether or not any user can visit a specified service as well as a 
     * "per user" table that controls whether a specific user can visit a specified
     * service. 
     * <p>
     * This class also Contains functions to enable/disable a service either for 
     * all users or a specified user as well as query information about the 
     * re-activation time of a specified service if it is disabled. 
     * <p>
     * The default implementation is done in-memory, but this could be re-written
     * to persist as part of a file, DB, cloud, etc. implementation.
     * 
     * @author Michael Coates (michael.coates .at. owasp.org) 
     *         <a href="http://www.aspectsecurity.com">Aspect Security</a>
     * @author John Melton (jtmelton .at. gmail.com)
     *         <a href="http://www.jtmelton.com/">jtmelton</a>
     * @since February 24, 2010
     */
    public class AppSensorServiceController
    {

        /** logger */
        private static readonly ASLogger logger = APPSENSOR.AsUtilities.GetLogger("AppSensorServiceController");

        /** Service storage */
        private static readonly ServiceStore serviceStore = APPSENSOR.ServiceStore;

        /**
         * Defines the ThreadLocalRequestURI to store the current request URI for this thread.
         * This is needed to store the requestURI on the thread local - since the Esapi request gets dropped.
         */
        private class ThreadLocalRequestURI : ThreadLocal<String>
        {

            public String getRequestURI()
            {
                return base.Value;
            }

            public String initialValue()
            {
                return null;
            }

            public void setRequestURI(String requestURI)
            {
                base.Value = requestURI;
            }
        }

        /**
         * Instance of the current request uri thread local
         */
        private static ThreadLocalRequestURI currentRequestURI = new ThreadLocalRequestURI();

        /**
         * Method to set current request URI
         * @param reqURI current request URI
         */
        public static void setCurrentRequestURI(String reqURI)
        {
            currentRequestURI.setRequestURI(reqURI);
        }

        /**
         * Method to get current request URI 
         * @return current request URI
         */
        public static String getCurrentRequestURI()
        {
            return currentRequestURI.getRequestURI();
        }

        /**
         * This method checks if a service (defaults to request URI) is active in general 
         * and for the specific user, and returns true if it's active, false if not
         * @param service name of the service to check
         * @return whether or not the service is active
         */
        public static bool IsServiceActive(String service)
        {
            //check if service is disabled for all users
            bool isServiceActiveForAllUsers = IsServiceActiveForAllUsers(service);

            //get current user, and check if service is disabled for just this user
            ASUser user = APPSENSOR.AsUtilities.CurrentUser;
            bool isServiceActiveForSpecificUser = IsServiceActiveForSpecificUser(service, user);

            //only say it's active if both checks pass
            if (isServiceActiveForAllUsers && isServiceActiveForSpecificUser)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /**
         * This method checks if the specified service has been disabled for 
         * all users in the system, and returns true if so, false if not
         * @param service name of the service to check
         * @return whether or not the service is active for all users
         */
        public static bool IsServiceActiveForAllUsers(String service)
        {
            bool isActive = true; // fail open by design - request should be allowed if it's not explicitly denied

            if (ShouldCheck(service))
            { // we don't care about most requests (images, css, etc)
                //get the monitor for this service (ie URI)
                AppSensorServiceMonitor serviceMon = serviceStore.GetService(service);

                // check isActive flag
                if (serviceMon.IsActive == false)
                {
                    logger.Debug("Service <" + service + "> is disabled, checking re-activation time");

                    //tag service as inactive
                    isActive = false;

                    //get current time and service reactivation time for comparison
                    var currentTime = DateTime.Now;
                    var reActiveTime = serviceMon.ReActivateTime;

                    // check reActivateTime to see if we've reached it
                    if (currentTime >= reActiveTime)
                    {
                        //if we've reached the reactivation time, re-enable and allow access
                        EnableService(service);
                        isActive = true;
                    }
                }
                else
                {
                    // service monitor indicates service is active, so allow access
                    isActive = true;
                }
            }
            return isActive;
        }

        /**
         * This method checks if the specified service has been disabled for 
         * the specified user, and returns true if so, false if not
         * @param service name of the service to check
         * @param user user to check 
         * @return whether or not the service is active for the specified user
         */
        public static bool IsServiceActiveForSpecificUser(String service, ASUser user)
        {
            bool isActive = true; // fail open by design - request should be allowed if it's not explicitly denied

            if (ShouldCheck(service))
            { // we don't care about most requests (images, css, etc)
                //get the current user
                String userId = user.AccountId.ToString();

                //get the monitor for this service (ie URI)
                AppSensorServiceMonitor serviceMon = serviceStore.GetServiceForUser(service, userId);

                // check isActive flag
                if (serviceMon.IsActive == false)
                {
                    logger.Debug("Service <" + service + "> is disabled for user <" + user.AccountName +
                            "> with id <" + userId + ">, checking re-activation time");

                    //tag service as inactive
                    isActive = false;

                    //get current time and service reactivation time for comparison
                    var currentTime = DateTime.Now;
                    var reActiveTime = serviceMon.ReActivateTime;

                    // check reActivateTime to see if we've reached it
                    if (currentTime >= reActiveTime)
                    {
                        //if we've reached the reactivation time, re-enable and allow access
                        EnableServiceForUser(service, userId);
                        isActive = true;
                    }
                }
                else
                {
                    // service monitor indicates service is active, so allow access
                    isActive = true;
                }
            }
            return isActive;
        }

        /**
         * This method checks the configuration files, and determines if the specified
         * resource should be checked for active/inactive.  This is determined by checking
         * if it exists in a list of named exceptions and/or if it's extension matches
         * an extension that should be checked.  It is checked only if it is not a named 
         * exception and if it's extension should be checked.
         * @param resourceID resource to possibly check, by default a request URI
         * @return whether or not the resource should be checked for active
         */
        private static bool ShouldCheck(String resourceID)
        {
            AppSensorSecurityConfiguration config = (AppSensorSecurityConfiguration)AppSensorSecurityConfiguration.GetInstance();

            // define list of exceptions to not check - white list that should always be accessible
            var exceptions = config.DisableComponentExceptions.ToList();

            //if the currently requested resource is whitelisted, allow access
            if (exceptions.Contains(resourceID))
            {
                logger.Debug("Ignoring service enabled check for " + resourceID + " since it is in the exceptions list");
                return false;
            }

            // define list of extensions to check - extensions that we may care to disable (ignores images, css, etc.)
            ICollection<String> extensionsToCheck = config.DisableComponentExtensionsToCheck.ToList() ;

            //loop through extensions comparing to our requested resource to see if we should check the service
            foreach (String extension in extensionsToCheck)
            {
                int length = extension.Length;
                int resourceLength = resourceID.Length;
                // compare last x characters of resourceID with extensionsToCheck. x=length of extensionsToCheck
                int subStringLength = resourceLength - length;
                if (subStringLength < 0)
                {
                    subStringLength = 0;
                }
                String resouceExtension = resourceID.Substring(subStringLength);
                if (resouceExtension.Equals(extension, StringComparison.OrdinalIgnoreCase))
                {
                    //only return true if our requested resource matches one of the extensions
                    return true;
                }
            }

            return false;
        }

        /**
         * Gets the re-activation time for a specified service.
         * Picks the greatest time of per-user disabling or overall disabling of the service.
         * @param service name of the service to check
         * @param user name of user to check
         * @return time specified as a long for the time to reactivate the specified service 
         */
        public static DateTime GetServiceReactivationTime(String service, ASUser user)
        {
            //get service disabled time for all users
            var serviceMonReactivateTime = GetServiceOnlyReactivationTime(service);

            //get service disabled time for specified user
            var serviceReactivationTimeForUser = GetServiceReactivationTimeForUser(service, user);

            //pick the largest (latest) time and return
            if (serviceMonReactivateTime > serviceReactivationTimeForUser)
            {
                return serviceMonReactivateTime;
            }
            else
            {
                return serviceReactivationTimeForUser;
            }
        }

        /**
         * Gets the re-activation time for a specified service
         * @param service name of the service to check
         * @return time specified as a long for the time to reactivate the specified service
         */
        private static DateTime GetServiceOnlyReactivationTime(String service)
        {
            AppSensorServiceMonitor serviceMon = serviceStore.GetService(service);
            return serviceMon.ReActivateTime;
        }

        /**
         * Gets the re-activation time for a specified service / user combination
         * @param service name of the service to check
         * @param user name of the user to check
         * @return time specified as a long for the time to reactivate the specified service 
         * 		for the specified user
         */
        private static DateTime GetServiceReactivationTimeForUser(String service, ASUser user)
        {
            String userId = user.AccountId.ToString();
            AppSensorServiceMonitor perUserServiceMon = serviceStore.GetServiceForUser(service, userId);
            return perUserServiceMon.ReActivateTime;
        }

        /**
         * Disables a specified service (page,function, etc.) for a specified duration (eg. 30 minutes)
         * @param service name of the service to disable
         * @param duration numeric portion of duration to disable, eg. 30
         * @param timeScale time scale portion of duration to disable, eg. "m" (s,m,h,d)
         */
        public static void DisableService(String service, int duration, TimeIncrement timeScale)
        {
            // does a AppSensorServiceMonitor object exist for this service?
            AppSensorServiceMonitor serviceMon = serviceStore.GetService(service);
            serviceMon.Disable(duration, timeScale);
            logger.Info("Disabled Service:" + service + " for " + duration + " " + timeScale);
            serviceStore.Save(serviceMon);
        }

        /**
         * Disables a specified service (page,function, etc.) for a specified duration (eg. 30 minutes)
         * for a specified user
         * @param service name of the service to disable
         * @param userId name of user the service is to be disabled for
         * @param duration numeric portion of duration to disable, eg. 30
         * @param timeScale time scale portion of duration to disable, eg. "m" (s,m,h,d)
         */
        public static void DisableServiceForUser(String service, String userId, int duration, TimeIncrement timeScale)
        {
            // does a AppSensorServiceMonitor object exist for this service?
            AppSensorServiceMonitor serviceMon = serviceStore.GetServiceForUser(service, userId);
            serviceMon.Disable(duration, timeScale);
            logger.Info("Disabled Service: <" + service +
                    "> for user <" + userId + "> for <" + duration + "> <" + timeScale + ">");
            serviceStore.Save(serviceMon);
        }

        /**
         * Enables the specified service for all users
         * @param service name of the service to enable
         */
        private static void EnableService(String service)
        {
            AppSensorServiceMonitor serviceMon = serviceStore.GetService(service);
            serviceMon.Enable();
            serviceStore.Save(serviceMon);
        }

        /**
         * Enables the specified service for the specified user
         * @param service name of the service to enable
         * @param userId name of the user the service is to be enabled for
         */
        private static void EnableServiceForUser(String service, String userId)
        {
            AppSensorServiceMonitor serviceMon = serviceStore.GetServiceForUser(service, userId);
            serviceMon.Enable();
            serviceStore.Save(serviceMon);
        }
    }
}